
-- 	Actor.hs

--	September 1999

--	This file contains the executable code for the theatrical case
-- 	study in Section 5 of `Modelling Reactive Multimedia: Events &
--	behaviours' by Helen Cameron, Peter King and Simon Thompson

--	More information about this and similar programs can be found at
--
--	  http://www.cs.ukc.ac.uk/people/staff/sjt/Fran

--	To use this program on your system you will most likely have to
--	modify the bitmap imports at the foot of this file, since they use
--	absolute pathnames. Note also that the `unix' style of pathnames
--	appears to be necessary.

module Actor where

import Fran

--  The top level definition in the scenario, pulling together the actor,
--  the phone and the background. It is shown in Fran by typing
--	displayU scenario
--  to the command prompt.

scenario :: User -> ImageB

scenario u
  = actor
      `over`
    phone
      `over`
    back

    where

--  The main components of the scenario are defined in a local definition (or
--  where clause) which takes up most of the module. Their interdependence is
--  reflected in their having interdependent definitions.

--  The actor is given by moving the actorImage along the part actorPath.
--  The actorImage is an ImageB; actorPath is a Vector2B, a varying two-
--  dimensional vector.

    actor :: ImageB

    actor
      = move actorPath actorImage

--  The path is built up of three parts. Initially the actor is in the start
--  position, startPos, downstage. Motion is stared by the event startMove.
--  He then moves upstage according to motion, until endMove occurs, when he
--  becomes and remains stationary upstage.

    actorPath :: Vector2B 

    actorPath 
       = startPos `untilEv` startMove $  
         motion   `untilEv` endMove $
         endPos

--  The movement of the actor given by motion is upwards, with yPos increasing
--  at a constant rate ...

    motion = vector2XY xPos yPos
    yPos   = yStart + atRateU 0.05		-- could also implement using 
						-- 0.05*(time - constantB (ust + 10))

--  ... until yPos is greater than or equal to the end position, yEnd.
--  At this point the event endMove happens.

    startMove = makeEvent [10]
    endMove   = predicateU (yPos >=* yEnd)

--  The image of the actor has three components: the man himself, the words
--  he speaks and the bubble against which they appear. The relative positioning
--  of the three remains the same (even when they move) and so they are taken to
--  be a single unit.

    actorImage :: ImageB

    actorImage
      = man 
          `over`
        wordsImage
          `over`
        bubbleImage

--  The words appear in blue, placed in a particular position.

    wordsImage :: ImageB

    wordsImage = moveXY 0.42 0.64 (withColor blue (stringBIm words))

--  The words are given by stepping through a number of different events,
--  using the function stepper. This function builds a step function with
--  values in type a from an Event a: if the event has an occurrence (t,x)
--  then the value of the step function becomes x at time t.

--  Initially there is a null word, then the line is delivered, one word
--  at a time, as given by the definition actorsLine. On starting to move
--  the word is again null, and on the end of the move the actor answers the
--  phone by saying "Hello".

    words 
      = stepper ""
        (actorsLine
         .|.
         startMove -=> ""
         .|.
         endMove -=> "Hello")

--  The line is given by associating the times 2.5, 3.5 etc. with the words
--  "Now", "is" and so forth.

    actorsLine 
      = makeEventVals
            [2.5,3.5 .. ] 
            ["Now","is","the","winter","of","our","discontent."]

--  The image of the bubble evolves in a similar way to the words.
--  untilEv is used to switch through a sequence of states.
  
    bubbleImage 
      = emptyImage `untilEv` makeEvent [1] $
        bubble     `untilEv` startMove     $
        emptyImage `untilEv` endMove       $
        bubble

--  The phone has three possible states: it is initally off, then it
--  rings and finally it is off the hook.

    phone :: ImageB
 
    phone = moveXY 0.6 0.7 $
            phoneOff     `untilEv` startMove $
            ringingPhone `untilEv` endMove   $
            liftedPhone

--  The ringing events are defined relative to the start of the move,
--  signalled by the startMove event. Here switcher is used. Initally the
--  phone is on: whenever ringOff occurs the phone goes off, and whenever
--  ringOn occurs it goes back on. 

--  The switcher function works in a similar way to stepper, except that it
--  builds arbitrary sorts of behaviours by sticking together the behaviours
--  returned in behaviour-valued events.

    ringOn, ringOff :: Event ()
    ringOn  = makeEventRelEvent startMove [2,4 .. 8]
    ringOff = makeEventRelEvent startMove [1,3 .. 9]

    ringingPhone 
      = phoneOn
         `switcher` 
          (ringOn  -=> phoneOn
           .|.
           ringOff -=> phoneOff)


--  Auxiliary functions ...


--  Auxiliary stuff factoring out all the uses of User into versions of
--  with the User already supplied. These come within the where clause.

    predicateU = flip predicate u	
    ust        = userStartTime u 	
    atRateU    = flip atRate u		
					
--  Auxiliary functions to make an (Event a) from a list of times and
--  a list of values and an Event () from a list of times.

    makeEventVals :: [Time] -> [a] -> Event a

    makeEventVals ts xs = possOccsE (zip (map (+ust) ts) (map Just xs))

    makeEvent :: [Time] -> Event ()

    makeEvent = makeEventRel ust


--  This is the end of the where clause defining the components of the 
--  scenario. In the remainder of the file are some auxiliary definitions
--  and constants, such as the bitmaps and the system geometry.

--  Two other auxiliary event-making functions. These don't use User
--  information and so are not included in the where clause.

--  makeEventRel makes an event from a list of occurence times plus
--  a starting time.

makeEventRel :: Double -> [Double] -> Event ()

makeEventRel t  
  = foldr (.|.) neverE . (map (timeIs . (t +)))


--  makeEventRelEvent is makeEventRel started at the first occurrence of
--  the event which is the first argument of the function.

makeEventRelEvent :: Event () -> [Double] -> Event ()

makeEventRelEvent ev ts
  = joinEOne (handleE ev (\t _ _ -> makeEventRel t ts))

--  Sequencing: this variant of the switcher is syntactically more
--  elegant, as well as separating the change event from the `after'
--  behaviour.

untilEv :: GBehavior b => b -> Event () -> b -> b

untilEv b1 ev b2 = b1 `switcher` (ev -=> b2)


--  ... and system constants.

--  The constant images, including (scaled) bitmap images and the background.
--  NB: you will need to make sure that these are modified in the appropriate
--  way when you use these files on your system.

phoneOn, phoneOff, base, handset, man, bubble :: ImageB

phoneOff    = importBitmap "c:/hugswork/Sept99/phone0.bmp"
phoneOn     = importBitmap "c:/hugswork/Sept99/phone1.bmp"
base        = importBitmap "c:/hugswork/Sept99/base.bmp"
handset     = moveXY 0 0.2 (importBitmap "c:/hugswork/Sept99/handset.bmp")
liftedPhone = handset `over` base          
man         = stretch 0.3 (importBitmap "c:/hugswork/Sept99/man.bmp")
bubble      = moveXY 0.4 0.6 (stretch 0.5 (importBitmap "c:/hugswork/Sept99/bubble.bmp"))

back = withColor (duller 0.05 yellow) (stretch 2.0 circle)

--  System geometry

xPos     = constantB (-0.4)
yStart   = constantB (-0.55)
yEnd     = constantB 0.45
startPos = vector2XY xPos yStart
endPos   = vector2XY xPos yEnd

