

--	Tickertape.hs

--	Showing the words arriving in an Event String in a ``tickertape''
--	style scrolling thing. Three different versions given here, with both
-- 	vertical and horizontal scrolling.

--	SJT 14.9.99


module Tickertape where

import Fran
import qualified StaticTypes as S		-- for Font etc.
import List					-- for intersperse

-- A test Event built from scratch. Words arrive once per second.

testInput :: Event String

testInput = possOccsE $ zip [1.0,2.0 .. ] (map Just words)
            where
            words = concat (repeat ["There","is","a","green","hill","far","away."])

-- A test Event built from scratch. Words arrive with differing frequency.

testInput2 :: Event String

testInput2 
  = possOccsE $ zip times (map Just words)
            where
            words = concat (repeat ["There","is","a","green","hill","far","away."])
            times = makeNums 10.0 [1.0,1.1,2.0,2.1,2.2,5.0,6.0,8.9,9.0,10.0]

-- Making an infinite list of numbers....

makeNums :: Num a => a -> [a] -> [a]

makeNums n ns = ns ++ makeNums n (map (+n) ns)

testInput3 :: Event String

testInput3 
  = possOccsE $ zip times (map Just words)
            where
            words = concat (repeat ["There","is","a","green","hill","far","away."])
            times = makeNums 20.0 [1.0,1.2,2.0,2.1,2.2,5.0,6.0,8.9,9.0,10.0]


-- The simplest way of showing some text: show the current word and displace it when 
-- the next one comes along.

testTest :: ImageB

testTest = showBIm (stepper "" testInput)

-- Testing showStringsPile ...

testPileDown :: ImageB

testPileDown = showStringsPileDown testInput 

testPileUp :: ImageB

testPileUp = showStringsPileUp testInput 

-- ... which shows the strings arriving in the event occurrences one 
-- below the other. The new words are lost at the bottom end of
-- the screen. Font selection appears not to work.

showStringsPileDown :: Event String -> ImageB

showStringsPileDown strEv
  = accumB transf emptyImage strImEv
    where
    courierIm = textImage . textFont S.system . constantB
    strImEv = strEv ==> stringIm
    transf beh img
      =   (moveXY 0.0 (-0.2) img 
           `over`
           beh)

-- The inner loop here is exactly the same as in showStringsPileDown,
-- but at the top level move 0.2 upwards at each event occurrence.
-- [Not the most elegant: should be able to have single control of the
-- position, rather than doing the accumulation and also doing a global
-- re-positioning.]

showStringsPileUp :: Event String -> ImageB

showStringsPileUp strEv
  = moveXY 0.0 upwards $
      accumB transf emptyImage strImEv
    where
    stringIm = textImage . textFont (S.courier) . simpleText . constantB
    strImEv = strEv ==> stringIm
    transf beh img
      =   (moveXY 0.0 (-0.2) img 
           `over`
           beh)
    upwards = stepper 0.0 (scanlE (+) 0.0 (strEv -=> 0.2)) 

-- Testing showBuffered ...

testBuffered, testBuffered2, testBuffered3 :: ImageB

testBuffered = showBuffered 10 testInput
testBuffered2 = showBuffered 10 testInput2

-- which uses a buffer containing size words. These are concatenated and 
-- shown centered; this works pretty well, but of course lose left-justification
-- as the total length of the string can vary from step to step.

-- It is left to think of how this might be done to give left-justified
-- text: as strings are shown in fixed width (courier) font, can do the 
-- appropriate arithmetic...

showBuffered :: Int -> Event String -> ImageB

showBuffered size strEv
  = stringIm bufferBeh
    where
    stringIm = textImage . textFont (S.courier) . simpleText 
    bufferBeh = stepper [] joinedBuffer
    joinedBuffer = buffer ==> (concat . intersperse " " . reverse)
    buffer = scanlE consChop [] strEv
    consChop xs x = take size (x:xs)

-- Padding out the intervals between event occurrences: buffering in 
-- some sense, but done by transforming an Event (i.e. stream of event 
-- occurrences) rather than by a lower-level, explicit storage, solution.
-- The numeric parameter is the minimum interval between event occurrences.

-- The top level function is padEvent ...

padEvent :: Double -> Event a -> Event a

padEvent minT ev
  = nextEventOcc 0 minT ev

nextEventOcc :: Double -> Double -> Event a -> Event a

--        ... but this uses nextEventOcc, whose extra parameter carries the
-- information about the next permissible time for an event occurrence to occur.

nextEventOcc nextT minT ev
  = joinEOne $
    ev `handleE` (\t v ev' ->  let t' = max nextT t in
				      possOccsE [(t',Just v)] 
                                         .|. 
 				      nextEventOcc (t'+minT) minT ev')

-- testing event padding in the buffer.

testBuffered3 = tickertape 10 1 testInput2
testBuffered4 = tickertape 10 1 testInput3

-- The tickertape function prints its input in - at steady state - a sliding
-- window of n words. Words appear with a minimum spacing of t seconds.

tickertape :: Int -> Double -> Event String -> ImageB

tickertape n t evStr 
  = showBuffered n (padEvent t evStr)

-- Delaying the occurrences of an Event by a given amount.

laterEv :: Double -> Event a -> Event a

laterEv delay ev
  = joinEOne $
    ev `handleE` (\t v ev' ->  possOccsE [(t+delay,Just v)] 
                                  .|. 
 			       laterEv delay ev')

-- How to add events in quiescent periods? A clock ticks every
-- gap seconds; it will insert an event occurrence containing x
-- if the last real event occurrence was at least gap seconds ago.
-- This is signalled by the count of `outstanding' event occurrences
-- being zero. On an event occurrence 1 is added to the count, and gap
-- seconds later one is subtracted.

addQuiescentEvents :: Double -> a -> Event a -> Event a

addQuiescentEvents gap x ev
  = ev .|. extras
    where
    extras = clock `whenE` isPending
    evUp = ev -=> 1
    evDown = (laterEv gap ev) -=> (-1)
    evChange = evUp .|. evDown
    pendingCount = accumB (+) 0 evChange
    isPending = (pendingCount ==* 0)
    clock = possOccsE (zip [gap,gap+gap .. ] (map Just (repeat x)))

-- Testing with quiescent events

testBuffered5 = showBuffered 10 (addQuiescentEvents 2 "." (padEvent 1 testInput3))

-- Top-level buffering and padding.

ticker :: Event String -> ImageB

ticker = showBuffered 10 . addQuiescentEvents 2 "." . padEvent 1


                                                                      
