||----------------------------------------------------------------------|| 
||									||
||	express.m							||
||									||
||	The type of integer expressions					||
||	and a parser for it.						||
||									||
||	16 June 1988							||
||									||
||----------------------------------------------------------------------|| 

%include "parse.m"

||----------------------------------------------------------------------|| 
||	First we define the syntax of the numerical expressions.	||
||									||
||	The concrete syntax is as follows:				||
||		Literals :	67,023,~89,~123 etc.			||
||		Variables :	a,b,c,...,z				||
||		Binary ops : 	+,*,-,/,% (last is for mod)		||
||	Expressions are fully bracketed, if compound, i.e.		||
||		(23+(34-45))						||
||	White space not permitted.					||
||									||
||	Now for the abstract syntax...					||
||----------------------------------------------------------------------|| 

var == char

op ::= Add | Sub | Mult | Div | Mod

expr ::= Lit num |
	 Var var |
	 Op op expr expr

||----------------------------------------------------------------------||
||	Then we define various auxilliary functions which recognise	||
||	various components of the concrete representations of the 	||
||	expressions.							||
||----------------------------------------------------------------------|| 

isVar :: char -> bool

isVar ch = ('a' <= ch <= 'z')

isOp  :: char -> bool

isOp = member [ '+' , '-' , '*' , '/' , '%' ]

isDigit :: char -> bool

isDigit ch = ('0' <= ch <= '9')

lb :: char

lb = '('

rb :: char

rb = ')'

||----------------------------------------------------------------------|| 
||	We now introduce some routines for conversion between 		||
||	concrete representations of numbers and operators and 		||
||	their abstract counterparts.					||
||									||
||	charlistToExpr takes a list of characters representing 	||
||	an integer, and returns (Lit n) where n is the value of the	||
||	string. The function uses..					||
||									||
||	diglistToNum which converts a list of digits into a		||
||	(positive) integer.						||
||									||
||	charToOp does the obvious thing.				||
||----------------------------------------------------------------------|| 

charlistToExpr :: [char] -> expr

charlistToExpr l
	= Lit ( (signL) * diglistToNum (numpartL) )
	  where
	  (signL,numpartL) = (1,[])	  , if  l=[]
			   = (1,l)	  , if  hd l ~= '~'
			   = (-1,tl l)  , otherwise


diglistToNum :: [char] -> num

diglistToNum 
     =  (foldl' add 0) . (map digitToNum) 
	where
	digitToNum ch = code ch - code '0'
	foldl' f st [] = st
	foldl' f st (a:x) = foldl' f (f st a) x
	add n d = n*10 + d

charToOp :: char -> op

charToOp '+' = Add
charToOp '-' = Sub
charToOp '*' = Mult
charToOp '/' = Div
charToOp '%' = Mod

||----------------------------------------------------------------------|| 
||	The parser itself. At the top level there are three		||
||	alternatives, which we'd expect from the form of the 		||
||	grammar.							||
||	The first recognises literals, the second variables, and	||
||	the third operator expressions.					||
||----------------------------------------------------------------------|| 


parser :: parse char expr

parser = litParse $alt varParse $alt opExpParse

||----------------------------------------------------------------------|| 
||	LitParse uses the optional constructor - it may or may not	||
||	recognise the token '~' (for unary minus). The result of this	||
||	is joined to the digit list following, and then conversion	||
||	takes place.							||
||----------------------------------------------------------------------|| 

litParse = ( ( (optional (token '~')) $then
	    	(list (spot isDigit)) 
	      ) $do join 
	    ) $do charlistToExpr
	     
		where
		join (l,m) = l++m

||----------------------------------------------------------------------|| 
||	Spot a variable, and then identify it as such, by tagging it	||
||	with Var.							||
||----------------------------------------------------------------------|| 

varParse = spot isVar $do Var

||----------------------------------------------------------------------|| 
||	Recognise, sequentially, a left bracket, an expression, an	||
||	operator, another expression and finally a right bracket.	||
||	This results in a nested sequence of pairs, which is unpacked	||
||	and reassembled by makeExpr.					||
||----------------------------------------------------------------------|| 

opExpParse 
      =	( token lb   $then
	  parser     $then
	  spot isOp $then
	  parser     $then
	  token rb ) $do
	  makeExpr
	  where
	  makeExpr (lb,(e1,(bop,(e2,rb)))) = Op (charToOp bop) e1 e2

||----------------------------------------------------------------------|| 
||	Commands have a very simple syntax, taking one of two		||
||	forms. An							||
||		expr							||
||	is simply evaluated, and					||
||		var:expr						||
||	causes the variable to be updated to have the expression value.	||
||									||
||	May 94: the Null command is included as the effect of a syntax	||
||	error.								||
||----------------------------------------------------------------------|| 

command ::= Eval expr | Assign var expr | Null

||----------------------------------------------------------------------|| 
||	Parsing commands - uses the parser for expressions in a		||
||	simple-minded extension.					||
||----------------------------------------------------------------------|| 

commandParse :: parse char command

commandParse = (parser $do Eval) $alt assignParse

assignParse
      = ( (spot isVar) $then
	  token ':'     $then 
	  parser ) $do
	  makeAssign
	  where
	  makeAssign (v,(':',e)) = Assign v e
