||----------------------------------------------------------------------||
||									||
||	sTreeAA.m							||
||									||
||	Binary search trees -- 						||
||   		i.e. sorted binary trees 				||
||		     without repetitions.				||
||	The trees are implemented as abstype, as in sTreeAb.m, but 	||
||	also have a size field added, as in sTreeAu.m. This shows that	||
||	less re-implementation is needed than in the case of sTreeAu.m	||
||									||
||	26 April 1994							||
||									||
||----------------------------------------------------------------------||

abstype tree *
with 
	delete :: * -> tree * -> tree *
	insert :: * -> tree * -> tree *
	isNil,isNode :: tree *  -> bool
	join :: tree * -> tree * -> tree *
	leftSub,rightSub :: tree * -> tree *
	nil :: tree *
	size :: tree * -> num
	treeVal :: tree * -> *
	tree1,tree2 :: tree num

||----------------------------------------------------------------------||
||	The implementation of trees.					||
||----------------------------------------------------------------------||

itree * ::= Nil | Node * num (itree *) (itree *)

tree * == itree *

nil = Nil

||----------------------------------------------------------------------||
||	Error types used -- import the declaration.			||
||----------------------------------------------------------------------||

%include "errorType"

||----------------------------------------------------------------------||
||	Constructing trees.						||
||	insert is idempotent.						||
||----------------------------------------------------------------------||

||	insert :: * -> tree * -> tree *

insert val Nil = (Node val 1 Nil Nil)

insert val (Node v n t1 t2)
	= Node v n t1 t2			, if v=val
	= Node v (n+1) t1 (insert val t2)	, if val > v
	= Node v (n+1) (insert val t1) t2	, if val < v

||----------------------------------------------------------------------||
||	Selectors.							||
||----------------------------------------------------------------------||

||	leftSub,rightSub :: tree * -> tree *

leftSub (Node v n t1 t2)  = t1
leftSub Nil		= error "leftSub"

rightSub (Node v n t1 t2) = t2
rightSub Nil		= error "rightSub"

||	treeVal :: tree * -> *

treeVal (Node v n t1 t2)  = v
treeVal Nil             = error "treeVal"

size (Node v n t1 t2) = n
size Nil = 0

||----------------------------------------------------------------------||
||	Discrimination.							||
||----------------------------------------------------------------------||

||	isNil,isNode :: tree *  -> bool

isNil x = (x = Nil)

isNode = (~).isNil

||----------------------------------------------------------------------||
||	Deletion							||
||	Handles the simple cases -- passes the operation down into	||
||	recursive calls on subtrees, or spots a Nil tree case.		||
||									||
||	This function assumes that the element present, so no need 	||
||	to check for presence. Could alternatively do nothing when	||
||	an element is absent.						||
||									||
||	The tricky case is handled by join.				||
||----------------------------------------------------------------------||

||	delete :: * -> tree * -> tree *

delete val (Node v (n+1) t1 t2)
	= Node v n (delete val t1) t2	, if val < v
	= Node v n t1 (delete val t2)	, if val > v
	= t1				, if t2 = Nil
	= t2				, if t1 = Nil
	= join t1 t2			, otherwise

||----------------------------------------------------------------------||
||	Join two trees together. It is known that the first is		||
||	smaller than the second (pairwise by elements), and than 	||
||	neither is Nil.							||
||									||
||	Make a new Node, whose value is the minimum of the larger tree,	||
||	whose left subtree is the smaller tree and whose right is the	||
||	larger with its minimum element removed.			||
||----------------------------------------------------------------------||

||	join :: tree * -> tree * -> tree *

join t1 t2 = Node mini s t1 newt2
	     where
	     mini = val (minTree t2)
	     newt2 = delete mini t2
	     s = size t1 + size t2

||----------------------------------------------------------------------||
||	Searching for a value.						||
||	Variants are possible: return a boolean, or an error value,	||
||	or a value of error type.					||
||----------------------------------------------------------------------||

search :: * -> tree * -> tree *

search val t   	= nil		, if isNil t
		= t 		, if val = v
	    	= search val t1	, if val < v
	    	= search val t2	, otherwise || val > v
	    	  where
	    	  v   = treeVal t
	    	  t1  = leftSub t
	    	  t2  = rightSub t

||----------------------------------------------------------------------||
||	Extremal values in a tree.					||
||----------------------------------------------------------------------||

maxTree :: tree * -> err *

maxTree t 	= Error		, if isNil t
		= OK v		, if isNil t2 
		= maxTree t2	, otherwise
		  where
		  t2 = rightSub t
		  v  = treeVal t

minTree :: tree * -> err *

minTree t 	= Error		, if isNil t
		= OK v		, if isNil t1 
		= minTree t1	, otherwise
		  where
		  t1 = leftSub t
		  v  = treeVal t

||----------------------------------------------------------------------||
||	Successor							||
||									||
||	This approach is quite different from the pointer-based one 	||
||	which appears in Cormen p249.					||
||									||
||	Note that can call for the successor of a non-element of the	||
|| 	tree; the effect is to find the smallest member of the tree	||
||	larger than the sought-after element.				||
||									||
||	Predecessor is dual.						||
||									||
||	Control flow here is hampered by the lack of exceptions; once	||
||	an acceptable value is found in the leftward calls, can		||
||	simply return it, without going through all nested calls.	||
||----------------------------------------------------------------------||

successor :: * -> tree * -> err *

successor v1 t 
	= Error				, if isNil t
	= successor v1 t2		, if v1 > v2
	= minTree t2			, if v1 = v2
	= OK v2				, if trySmaller = Error
	= trySmaller			, otherwise
	  where
	  trySmaller = successor v1 t1
	  v2 = treeVal t
	  t1 = leftSub t
	  t2 = rightSub t

||----------------------------------------------------------------------||
||	Searching for the closest element in a tree.			||
||	Can this be related to loose matching of (e.g.) strings.	||
||									||
||	Assumes called on a non-empty tree, therefore a result exists.	||
||	Chooses the larger when there is a choice of 2 closest values.	||
||----------------------------------------------------------------------||

closest :: num -> tree num -> num

closest val t
	= v			, if val = v \/
				     (val < v & isNil t1) \/
				     (val > v & isNil t2)
	= closer val v1 v	, if val < v
	= closer val v v2	, if val > v
	  where
	  v = treeVal t
	  t1 = leftSub t
	  t2 = rightSub t
	  v1 = closest val t1
	  v2 = closest val t2
	  closer a b c = b 	, if c-a > a-b
	  	       = c	, otherwise

||----------------------------------------------------------------------||
||	Indexing -- find the ith element.				||
||----------------------------------------------------------------------||

indexT :: num -> tree * -> *

indexT n t
	= error "indexT"	, if isNil t
	= indexT n t1		, if n < st1
	= v			, if n = st1
	= indexT (n-st1-1) t2	, otherwise
	  where
	  v = treeVal t
	  t1 = leftSub t
	  t2 = rightSub t
	  st1 = size t1

||----------------------------------------------------------------------||
||	Examples							||
||----------------------------------------------------------------------||

tree1 = itree1

itree1 = Node 14 7 (Node 7 2 Nil (Node 9 1 Nil Nil)) 
		(Node 32 4 (Node 17 2 (Node 15 1 Nil Nil) Nil) (Node 49 1 Nil Nil))

tree2 = insert 16 tree1
