||----------------------------------------------------------------------||
||									||
||	sTreeAu.m							||
||									||
||	Binary search trees -- 						||
||   		i.e. sorted binary trees 				||
||		     without repetitions.				||
||	Additional SIZE information is given at the nodes.		||
||									||
||	This is an interesting example to look at in detail. How much	||
||	has to be modified in moving to the augmentted structure? In	||
||	this case nearly the whole file, since all the pattern 		||
||	matching is invalidated.					||
||									||
||	Feedback: the modification was nothing more than routine,	||
||	although it demanded re-definition of many of the functions	||
||	over the tree type. It breaks the principle that you should 	||
||	be able to augment functionality without providing additional	||
||	information to the user: information hiding, in other words.	||
||									||
||	This augmentation can also be seen as an exercise in 		||
||	memoisation, where the size operation (trivial recursive	||
||	function is memoised in place, as it were.			||
||									||
||	26 April 1994							||
||									||
||----------------------------------------------------------------------||

tree * ::= Nil | Node * num (tree *) (tree *)

||----------------------------------------------------------------------||
||	Error types used -- import the declaration.			||
||----------------------------------------------------------------------||

%include "errorType"

||----------------------------------------------------------------------||
||	Constructing trees.						||
||	insert is idempotent.						||
||----------------------------------------------------------------------||

insert :: * -> tree * -> tree *

insert val Nil = (Node val 1 Nil Nil)

insert val (Node v n t1 t2)
	= Node v n t1 t2			, if v=val
	= Node v (n+1) t1 (insert val t2)	, if val > v
	= Node v (n+1) (insert val t1) t2	, if val < v

||----------------------------------------------------------------------||
||	Deletion							||
||	Handles the simple cases -- passes the operation down into	||
||	recursive calls on subtrees, or spots a Nil tree case.		||
||									||
||	This function assumes that the element present, so no need 	||
||	to check for presence. Could alternatively do nothing when	||
||	an element is absent.						||
||									||
||	The tricky case is handled by join.				||
||----------------------------------------------------------------------||

delete :: * -> tree * -> tree *

delete val (Node v (n+1) t1 t2)
	= Node v n (delete val t1) t2	, if val < v
	= Node v n t1 (delete val t2)	, if val > v
	= t1				, if t2 = Nil
	= t2				, if t1 = Nil
	= join t1 t2			, otherwise

||----------------------------------------------------------------------||
||	Join two trees together. It is known that the first is		||
||	smaller than the second (pairwise by elements), and than 	||
||	neither is Nil.							||
||									||
||	Make a new Node, whose value is the minimum of the larger tree,	||
||	whose left subtree is the smaller tree and whose right is the	||
||	larger with its minimum element removed.			||
||----------------------------------------------------------------------||

join :: tree * -> tree * -> tree *

join t1 t2 = Node mini count t1 newt2
	     where
	     mini = val (minTree t2)
	     newt2 = delete mini t2
	     count = size t1 + size t2

||----------------------------------------------------------------------||
||	Extract the size component.					||
||----------------------------------------------------------------------||

size :: tree * -> num

size Nil = 0
size (Node v n t1 t2) = n


||----------------------------------------------------------------------||
||	Searching for a value.						||
||	Variants are possible: return a boolean, or an error value,	||
||	or a value of error type.					||
||----------------------------------------------------------------------||

search :: * -> tree * -> tree *

search val Nil               = Nil
search val (Node v n t1 t2)  = (Node v n t1 t2)	, if val = v
			     = search val t1	, if val < v
			     = search val t2	, otherwise || val > v

||----------------------------------------------------------------------||
||	Extremal values in a tree.					||
||----------------------------------------------------------------------||

maxTree :: tree * -> err *

maxTree Nil 	= Error
maxTree (Node v n t1 t2)
	= OK v		, if t2 = Nil
	= maxTree t2	, otherwise

minTree :: tree * -> err *

minTree Nil 	= Error
minTree (Node v n t1 t2)
	= OK v		, if t1 = Nil
	= minTree t1	, otherwise

||----------------------------------------------------------------------||
||	Successor							||
||									||
||	This approach is quite different from the pointer-based one 	||
||	which appears in Cormen p249.					||
||									||
||	Note that can call for the successor of a non-element of the	||
|| 	tree; the effect is to find the smallest member of the tree	||
||	larger than the sought-after element.				||
||									||
||	Predecessor is dual.						||
||									||
||	Control flow here is hampered by the lack of exceptions; once	||
||	an acceptable value is found in the leftward calls, can		||
||	simply return it, without going through all nested calls.	||
||----------------------------------------------------------------------||

successor :: * -> tree * -> err *

successor v1 Nil = Error

successor v1 (Node v2 n t1 t2)
	= successor v1 t2		, if v1 > v2
	= minTree t2			, if v1 = v2
	= OK v2				, if trySmaller = Error
	= trySmaller			, otherwise
	  where
	  trySmaller = successor v1 t1

||----------------------------------------------------------------------||
||	Searching for the closest element in a tree.			||
||	Can this be related to loose matching of (e.g.) strings.	||
||									||
||	Assumes called on a non-empty tree, therefore a result exists.	||
||	Chooses the larger when there is a choice of 2 closest values.	||
||----------------------------------------------------------------------||

closest :: num -> tree num -> num

closest val (Node v n t1 t2)
	= v			, if val = v \/
				     (val < v & t1 = Nil) \/
				     (val > v & t2 = Nil)
	= closer val v1 v	, if val < v
	= closer val v v2	, if val > v
	  where
	  v1 = closest val t1
	  v2 = closest val t2
	  closer a b c = b 	, if c-a > a-b
	  	       = c	, otherwise

||----------------------------------------------------------------------||
||	Indexing the set -- find the ith element in the ordering.	||
||	Assume i is a valid index.					||
||	In a tree with i elements, number them 0..i-1.			||
||----------------------------------------------------------------------||

indexT :: num -> tree * -> *

indexT k Nil = error "indexT"

indexT n (Node v k t1 t2)
	= indexT n t1		, if n < st1
	= v			, if n = st1
	= indexT (n-st1-1) t2	, otherwise
	  where
	  st1 = size t1

||----------------------------------------------------------------------||
||	Examples							||
||----------------------------------------------------------------------||

tree1 = Node 14 7 (Node 7 2 Nil (Node 9 1 Nil Nil)) 
		(Node 32 4 (Node 17 2 (Node 15 1 Nil Nil) Nil) (Node 49 1 Nil Nil))

tree2 = insert 16 tree1
